/****************************************************************************

    PROGRAM: PM.C

    PURPOSE: Demonstrates the DPMI functions for interfacing with real-mode
    device drivers.

****************************************************************************/
#include <dos.h>
#include <stdio.h>
#include <malloc.h>
#include <windows.h>
#include "dpmi.h"
#include "pm.h"

RM386_INT rmInt;                        //structure for making real-mode
                                        //interrupts from protected mode
SELECTOR PMSelector;                    //protected mode selector
SEGMENT  RMSegment;                     //real-mode segment
DWORD    dwSegSelector;                 //seg:selector pair returned by
char     szOutputBuf[]= "RMDRIVER: This string is from the application.";
char     szMessage[100];                //message line
union    REGS inregs, outregs;          //holds registers for int21 calls
struct   SREGS sregs;                   //holds segment regs for int21 calls
#define  BUFFER_SIZE  100L              //size of transfer buffer
#define  TEXT_INC     15                //space between lines of text
int      pmTextY = 0;                   //Y coord of output line

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

****************************************************************************/
int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                            /* current instance             */
HANDLE hPrevInstance;                        /* previous instance            */
LPSTR lpCmdLine;                             /* command line                 */
int nCmdShow;                                /* show-window type (open/icon) */
{
    MSG msg;                                 /* message                      */

    if (!hPrevInstance)                  /* Other instances of app running? */
        if (!InitApplication(hInstance)) /* Initialize shared things */
            return (FALSE);              /* Exits if unable to initialize     */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */
    while (GetMessage(&msg,        /* message structure                      */
            NULL,                  /* handle of window receiving the message */
            NULL,                  /* lowest message to examine              */
            NULL))                 /* highest message to examine             */
        {
        TranslateMessage(&msg);    /* Translates virtual key codes           */
        DispatchMessage(&msg);     /* Dispatches message to window           */
    }
    return (msg.wParam);           /* Returns the value from PostQuitMessage */
}
/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

****************************************************************************/
BOOL InitApplication(hInstance)
HANDLE hInstance;                              /* current instance           */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon =         NULL;
    wc.hCursor =       LoadCursor(NULL,IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  (LPSTR) "main";  /* Name of menu resource in .RC file. */
    wc.lpszClassName = "DPMI";          /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}
/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

****************************************************************************/
BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "DPMI"         ,                /* See RegisterClass() call.          */
        "PM to RM Interface Program",   /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW,            /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    return (TRUE);               /* Returns the value from PostQuitMessage */

}
/****************************************************************************

    FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages

****************************************************************************/
long FAR PASCAL MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                                /* window handle                   */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* additional information          */
LONG lParam;                              /* additional information          */
{
HDC  hDC;

    switch (message)
    {
        case WM_CREATE:
        {
        WORD wFlags=0, wMajorVer=0, wMinorVer=0, wProcessor=0;
            if ( !(GetWinFlags() & WF_PMODE) ||
                !dpmiVersion (&wFlags, &wMajorVer, &wMinorVer, &wProcessor))
            {
                MessageBox(hWnd,"DPMI services unavailable","PMAPP",MB_OK);
                PostQuitMessage(0);
            }  //End if (not protected mode, or dpmi not available)
            else
            {
            char szPrompt[80];
                if (wProcessor==2) wProcessor=286;
                if (wProcessor==3) wProcessor=386;
                if (wProcessor==4) wProcessor=486;
                if (GetWinFlags()&WF_STANDARD)
                {
                    sprintf (szPrompt, "Standard Mode DPMI ver. %x.%x: Processor %d",
                             wMajorVer, wMinorVer, wProcessor);
                } //End if (standard mode)
                else
                {
                    sprintf (szPrompt, "Enhanced Mode DPMI ver. %d.%d: Processor %d",
                             wMajorVer, wMinorVer, wProcessor);
                } //End if (enhanced mode)
                MessageBox(hWnd, szPrompt,
                "PMAPP",MB_OK);
            } //End if (protected mode and dpmi available)
            break;
        } //End case (WM_CREATE)
        case WM_COMMAND:           /* message: command from application menu */
            switch (wParam)
              {
              case RW:
                  pmiCallDD (hDC = GetDC(hWnd));
                  ReleaseDC (hWnd, hDC);
                  break;
              case INT86:
                  pmiInt6b (hDC = GetDC(hWnd));
                  ReleaseDC (hWnd, hDC);
                  break;
              case DPMI:
                  pmiDPMIInt6b (hDC = GetDC(hWnd));
                  ReleaseDC (hWnd, hDC);
                  break;
              case DDATA:
                  pmiGetDriverData (hDC = GetDC(hWnd));
                  ReleaseDC (hWnd, hDC);
                  break;
              default:                            /* Lets Windows process it       */
                return (DefWindowProc(hWnd, message, wParam, lParam));
                break;
              }
            break;
        case WM_DESTROY:                  /* message: window being destroyed */
            PostQuitMessage(0);
            break;

        default:                          /* Passes it on if unproccessed    */
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}
/****************************************************************************

    FUNCTION:  pmiCallDD

    PURPOSE:  Calls real-mode device driver, swaps data using a transfer buffer

****************************************************************************/
int pmiCallDD(HDC hDC)
{
LPSTR    lpTransferBuf;                 //points to transfer buffer
HANDLE   hDriver;                       //handle to driver
OFSTRUCT OfFileStruct;                  //structure for opening file

    sprintf (szMessage, "%s", "Read/Write Driver");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
    // Allocate the transfer buffer
    // Take selector from low word of return value
    // (real-mode segment is in high word)
    if ((dwSegSelector = GlobalDosAlloc (BUFFER_SIZE))==NULL)
    {
        return -1;
    } //End if (could not allocate transfer buf)
    PMSelector = LOWORD (dwSegSelector);
    RMSegment  = HIWORD (dwSegSelector);
    // Set pointer to transfer buffer
    lpTransferBuf = (LPSTR) MAKELONG (0, PMSelector);

    // Get a handle to the device driver
    if ((hDriver = OpenFile ("RMDRIVER", &OfFileStruct, OF_READWRITE))==-1)
    {
        sprintf (szMessage, "Error opening driver");
        TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
        return -1;
    } //End if (error opening driver)

    //Set raw io mode
    pmiSetRawMode (hDriver);

    // Read data from device driver
    inregs.x.ax = 0x3F00;      //Function to read from device
    inregs.x.bx = hDriver;     //Handle to device driver
    inregs.x.cx = 100;         //Bytes to read
    sregs.ds =    PMSelector;  //Protected mode address of transfer buffer
    inregs.x.dx = 0;           //Offset is 0 since buffer is paragraph aligned
    sprintf (szMessage, "Buffer selector:offset %Fp  (segment %x)",
             lpTransferBuf, RMSegment);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    //Call DOS int 21 services
    intdosx (&inregs, &outregs, &sregs);

    // Print the buffer;  it now contains data from the device
    //driver.
    lstrcpy ((LPSTR) szMessage, lpTransferBuf);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));

    // Write string to device driver
    lstrcpy (lpTransferBuf, (LPSTR) szOutputBuf);
    inregs.x.ax = 0x4000;      //Function to write to device
    inregs.x.bx = hDriver;     //Handle to the driver
    inregs.x.cx = 100;         //Bytes to write
    sregs.ds =    PMSelector;  //Protected mode address of transfer buffer
    inregs.x.dx = 0;           //Offset is 0 since buffer is paragraph aligned
    intdosx (&inregs, &outregs, &sregs);

    GlobalDosFree (PMSelector);

    // Close the device driver
    if (_lclose (hDriver)!=0)
    {
        sprintf (szMessage, "Error closing driver");
        TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
        return -1;
    } //End if (error closing driver)

    return 0;
} //End function (pmiCallDD)
/****************************************************************************

    FUNCTION:  pmiInt6b

    PURPOSE:  Performs int6b using int86x

    COMMENTS:


****************************************************************************/
int pmiInt6b(HDC hDC)
{
    sprintf (szMessage, "%s", "Int86x 6Bh");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
    // Allocate the transfer buffer
    // Take selector from low word of return value
    // (real-mode segment is in high word)
    if ((dwSegSelector = GlobalDosAlloc (BUFFER_SIZE))==NULL)
    {
        return -1;
    } //End if (could not allocate transfer buf)
    PMSelector = LOWORD (dwSegSelector);
    RMSegment  = HIWORD (dwSegSelector);
    sprintf (szMessage, "Selector: %x Segment: %x", PMSelector, RMSegment);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));

    inregs.x.ax =  0x1001;
    inregs.x.bx =  0x1001;
    inregs.x.cx =  0x1001;
    inregs.x.dx =  0x1001;
    inregs.x.si =  0x1001;
    inregs.x.di =  0x1001;
    sregs.ds =     PMSelector;
    sregs.es =     PMSelector;
    sprintf (szMessage, "Registers");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "AX, BX, CX, DX: %x, %x, %x, %x", inregs.x.ax,
             inregs.x.bx, inregs.x.cx, inregs.x.dx);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "SI, DI: %x, %x", inregs.x.si, inregs.x.di);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "DS, ES: %x, %x", sregs.ds, sregs.es);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    int86x (0x6b, &inregs, &outregs, &sregs);

    GlobalDosFree (PMSelector);

    return 0;
} //End function (pmiInt6b)
/****************************************************************************

    FUNCTION:  pmiSetRawMode

    PURPOSE:  Places the device driver in raw (binary) mode

    COMMENTS:


****************************************************************************/
int pmiSetRawMode(HANDLE hDriver)
{
    // Set the dd to raw mode - get, then set device control bits
    //bx - file handle for the driver
    //ax - 0x4400  Function for IOCTL call to read device control bits
    inregs.x.bx = hDriver;
    inregs.x.ax = 0x4400;
    intdos (&inregs, &outregs);
    // Now set raw mode bit of the device control bits
    //dl - bit 0x20 is set to indicate DOS should use raw mode with this
    //     device.  All other bits are unchanged.
    //ax - 0x4401   IOCTL function to set device bits
    inregs.x.bx = hDriver;
    inregs.h.dh=0;
    inregs.h.dl= outregs.h.dl | 0x20;
    inregs.x.ax = 0x4401;
    intdos (&inregs, &outregs);

    return 0;
} //End function (pmiSetRawMode)
/****************************************************************************

    FUNCTION:  pmiDPMIInt6b

    PURPOSE:  Performs int6b using DPMI

    COMMENTS:


****************************************************************************/
int pmiDPMIInt6b(HDC hDC)
{
    sprintf (szMessage, "%s", "DPMI Int 6Bh");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
    // Allocate the transfer buffer
    // Take selector from low word of return value
    // (real-mode segment is in high word)
    if ((dwSegSelector = GlobalDosAlloc (BUFFER_SIZE))==NULL)
    {
        return -1;
    } //End if (could not allocate transfer buf)
    PMSelector = LOWORD (dwSegSelector);
    RMSegment  = HIWORD (dwSegSelector);
    sprintf (szMessage, "Selector: %x Segment: %x", PMSelector, RMSegment);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));

    rmInt.edi = 0x1001;
    rmInt.esi = 0x1001;
    rmInt.ebx = 0x1001;
    rmInt.edx = 0x1001;
    rmInt.ecx = 0x1001;
    rmInt.eax = 0x1001;
    rmInt.es =  PMSelector;
    rmInt.ds =  PMSelector;
    sprintf (szMessage, "Registers");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "AX, BX, CX, DX: %lx, %lx, %lx, %lx", rmInt.eax,
             rmInt.ebx, rmInt.ecx, rmInt.edx);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "SI, DI: %lx, %lx", rmInt.esi, rmInt.edi);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    sprintf (szMessage, "DS, ES: %x, %x", rmInt.ds, rmInt.es);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    dpmiRMInt (0x6b, 0, &rmInt, 0);

    GlobalDosFree (PMSelector);

    return 0;
} //End function (pmiDPMIInt6b)
/****************************************************************************

    FUNCTION:  pmiGetDriverData

    PURPOSE:  Retrieves data from the device driver segment

    COMMENTS:


****************************************************************************/
int  pmiGetDriverData(HDC hDC)
{
LPSTR lpDriverData;
LPSTR lpDriverData2;
    sprintf (szMessage, "%s", "Get driver data");
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
    dpmiRMInt (0x6a, 0, &rmInt, 0);
    sprintf (szMessage, "RM Pointer to driver data %x:%x", rmInt.es,
             rmInt.ebx);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage, strlen(szMessage));
    if (!dpmiSegmentToSelector(rmInt.es, rmInt.ebx + 500, &PMSelector))
    {
        return FAILURE;
    }  //End if (error getting access to driver segment)
    lpDriverData = (LPSTR) MAKELONG (rmInt.ebx, PMSelector);
    sprintf (szMessage, "PM pointer to driver data %Fp",
             lpDriverData);
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) szMessage,
                 strlen(szMessage));
    TextOut (hDC, 5, pmTextY+=TEXT_INC, (LPSTR) lpDriverData, lstrlen(lpDriverData));
    //Always free selectors when done with them
    dpmiFreeSelector (PMSelector);
    return SUCCESS;
} //End (pmiGetDriverData)
