/*****************************************************/
/* bench.c                                           */
/* -- Benchmark various GDI calls.                   */
/* -- To build:                                      */
/*    "cc -O -DSTRICT bench.c fastline.c bench.rc    */
/*     mmsystem.lib"                                 */
/*****************************************************/

/*****************************************************/
/* Header files.                                     */
/*****************************************************/
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include "fastline.h"
#include "bench.h"

/*****************************************************/
/* Types.                                            */
/*****************************************************/
typedef struct
    {
    BOOL    fStraight;  /* Straight or diagonal? */
    UINT    dxPen;      /* Width of pen. */
    UINT    cact;       /* # iterations. */
    } LBM;  /* Line benchmark metrics. */

/*****************************************************/
/* Globals.                                          */
/*****************************************************/
char    szClass[]   = "BenchClass";
char    szTitle[]   = "GDI Benchmarks";

/* Times for rectangle operations. */
DWORD   dtimExtTextOut, dtimPatBlt;

/* Times for thick pen lines. */
DWORD   dtimPenLine, dtimFastLine;

/* Times for aligned/unaligned draws. */
DWORD   dtimAligned, dtimUnaligned;

/* Colors to cycle through. */
COLORREF    rgclr[] =
    {
    RGB(0x00, 0x00, 0x00),
    RGB(0x00, 0x00, 0xff),
    RGB(0x00, 0xff, 0x00),
    RGB(0x00, 0xff, 0xff),
    RGB(0xff, 0x00, 0x00),
    RGB(0xff, 0x00, 0xff),
    RGB(0xff, 0xff, 0x00),
    RGB(0xff, 0xff, 0xff)
    };

/* Default line benchmark metrics. */
LBM lbm = { FALSE, 6, 200 };

BOOL    fInBench;   /* Benchmark in progress? */

/*****************************************************/
/* Constants.                                        */
/*****************************************************/
#define cclr   (sizeof rgclr / sizeof rgclr[0])
#define crect  1000 /* # of rects in benchmk. */

/*****************************************************/
/* Prototypes.                                       */
/*****************************************************/
/* Exported. */
LRESULT CALLBACK __export
  LwWndProc(HWND hwnd, UINT wm, WPARAM wParam,
    LPARAM lParam);
BOOL CALLBACK __export
  FLineMetricsDlgProc(HWND hwnd, UINT wm,
    WPARAM wParam, LPARAM lParam);

/* Private. */
void BenchLines(HWND hwnd);
void BenchPlacement(HWND hwnd);
void BenchRectangles(HWND hwnd);
BOOL FGetW(UINT * puw, UINT did, HWND hwnd);
void PaintWnd(HWND hwnd);
void PenLine(HDC hdc, POINT rgpt[2], int dxThick,
  COLORREF clr);
int  YWriteTextLine(HDC hdc, int x, int y,
  TEXTMETRIC * ptxm, char * psz);

/*****************************************************/
/* Routines.                                         */
/*****************************************************/
int PASCAL
WinMain(HINSTANCE hins, HINSTANCE hinsPrev,
  LPSTR lpszCmd, int wShow)
/*****************************************************/
/* -- Entry point.                                   */
/*****************************************************/
    {
    MSG     msg;
    HWND    hwnd;

    if (hinsPrev == NULL)
        {
        WNDCLASS    wcs;

        wcs.style = CS_BYTEALIGNCLIENT;
        wcs.lpfnWndProc = LwWndProc;
        wcs.cbClsExtra = 0;
        wcs.cbWndExtra = 0;
        wcs.hInstance = hins;
        wcs.hIcon = LoadIcon(NULL, IDI_APPLICATION);
        wcs.hCursor = LoadCursor(NULL, IDC_ARROW);
        wcs.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
        wcs.lpszMenuName = MAKEINTRESOURCE(mnuBench);
        wcs.lpszClassName = szClass;
        if (!RegisterClass(&wcs))
            return FALSE;
        }

    msg.wParam = 0;
    if ((hwnd = CreateWindow(szClass, szTitle,
      WS_OVERLAPPEDWINDOW, CW_USEDEFAULT,
      CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
      NULL, NULL, hins, NULL)) != NULL)
        {
        ShowWindow(hwnd, wShow);
        while (GetMessage(&msg, NULL, 0, 0))
            {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
            }
        }
    return msg.wParam;
    }

LRESULT CALLBACK __export
LwWndProc(HWND hwnd, UINT wm, WPARAM wParam,
  LPARAM lParam)
/*****************************************************/
/* -- Main window procedure.                         */
/*****************************************************/
    {
    switch (wm)
        {
    default:
        break;

    case WM_DESTROY:
        PostQuitMessage(0);
        break;

    case WM_PAINT:
        PaintWnd(hwnd);
        return 0;

    case WM_COMMAND:
        switch (wParam)
            {
        default:
            break;

        case idmRectangles:
            BenchRectangles(hwnd);
            return 0;

        case idmLines:
            BenchLines(hwnd);
            return 0;

        case idmPlacement:
            BenchPlacement(hwnd);
            return 0;

        case idmControlLines:
            DialogBox(GetWindowInstance(hwnd),
              MAKEINTRESOURCE(dlgLines), hwnd,
              FLineMetricsDlgProc);
            break;
            }   /* End switch wParam. */
        break;  /* End case WM_COMMAND. */
        }       /* End switch wm. */

    return DefWindowProc(hwnd, wm, wParam, lParam);
    }

void
PaintWnd(HWND hwnd)
/*****************************************************/
/* -- Handle a WM_PAINT for the main window.         */
/* -- hwnd : Main window.                            */
/*****************************************************/
    {
    PAINTSTRUCT wps;
    char        szBuf[256];
    TEXTMETRIC  txm;
    int         y   = 0;

    if (BeginPaint(hwnd, &wps) == NULL || fInBench)
        goto LCleanup;
    GetTextMetrics(wps.hdc, &txm);

    /* Rectangle result. */
    y = YWriteTextLine(wps.hdc, 0, y, &txm,
      "Rectangles:");
    wsprintf(szBuf, "ExtTextOut=%ld", dtimExtTextOut);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);
    wsprintf(szBuf, "PatBlt=%ld", dtimPatBlt);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);

    /* Line results. */
    y = YWriteTextLine(wps.hdc, 0, y, &txm, "Lines");
    wsprintf(szBuf, "Pen lines=%ld", dtimPenLine);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);
    wsprintf(szBuf, "Fast lines=%ld", dtimFastLine);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);

    /* Aligned/unaligned results. */
    y = YWriteTextLine(wps.hdc, 0, y, &txm,
      "Alignment");
    wsprintf(szBuf, "Aligned=%ld", dtimAligned);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);
    wsprintf(szBuf, "Unaligned=%ld", dtimUnaligned);
    y = YWriteTextLine(wps.hdc, 20, y, &txm, szBuf);

LCleanup:
    EndPaint(hwnd, &wps);
    }

int
YWriteTextLine(HDC hdc, int x, int y,
  TEXTMETRIC * ptxm, char * psz)
/*****************************************************/
/* -- Write a null terminated string to the DC.      */
/* -- Return the vertical position for the next.     */
/* -- hdc  : Device context to write to.             */
/* -- x, y : Beginning of line.                      */
/* -- ptxm : Text metrics of current font.           */
/* -- psz  : String to write.                        */
/*****************************************************/
    {
    TextOut(hdc, x, y, psz, lstrlen(psz));
    return y + ptxm->tmHeight;
    }

void
BenchRectangles(HWND hwnd)
/*****************************************************/
/* -- Time ExtTextOut() vs PatBlt().                 */
/*****************************************************/
    {
    int         irect;
    DWORD       tim1, tim2;
    RECT        rect, rgrect[10];
    HDC         hdc;

    if ((hdc = GetDC(hwnd)) == NULL)
        return;

    /* Precompute rectangles to fill. */
    GetClientRect(hwnd, &rect);
    for (irect = 0; irect < 10; irect++)
        {
        rgrect[irect] = rect;
        InflateRect(rgrect + irect,
          MulDiv(irect, -rect.right, 20),
          MulDiv(irect, -rect.bottom, 20));
        }

    /* Begin benchmark. */
    tim1 = timeGetTime();
    for (irect = 0; irect < crect; irect++)
        {
        HBRUSH  hbrs, hbrsSav;
        RECT *  prect   = rgrect + (irect % 10);

        if ((hbrs = CreateSolidBrush(
          rgclr[irect % cclr])) != NULL)
            {
            hbrsSav = SelectObject(hdc, hbrs);
            PatBlt(hdc, prect->left, prect->top,
              prect->right - prect->left,
              prect->bottom - prect->top, PATCOPY);
            if (hbrsSav)
                SelectObject(hdc, hbrsSav);
            DeleteObject(hbrs);
            }
        }
    dtimPatBlt = (tim2 = timeGetTime()) - tim1;

    /* Do ExtTextOut() benchmark. */
    for (irect = 0; irect < crect; irect++)
        {
        COLORREF    clrSav;

        clrSav = SetBkColor(hdc, rgclr[irect % cclr]);
        ExtTextOut(hdc, 0, 0, ETO_OPAQUE,
          rgrect + (irect % 10), NULL, 0, NULL);
        SetBkColor(hdc, clrSav);
        }
    dtimExtTextOut = timeGetTime() - tim2;
    ReleaseDC(hwnd, hdc);
    InvalidateRect(hwnd, NULL, TRUE);
    }

void
BenchLines(HWND hwnd)
/*****************************************************/
/* -- Time MoveTo(),LineTo() with a thick pen vs.    */
/*    Rectangle().                                   */
/*****************************************************/
    {
    POINT   rgrgpt[10][2];
    UINT    irgpt, ilin, ilinLim;
    RECT    rect;
    HDC     hdc;
    DWORD   tim;
    int     dx, dy;

    if ((hdc = GetDC(hwnd)) == NULL)
        return;

	fInBench = TRUE;
    /* Initialize lines to draw. */
    GetClientRect(hwnd, &rect);
    dx = 1 + ((int)lbm.dxPen + 1) / 2;
    InflateRect(&rect, -dx, -dx);
    dx = rect.right - rect.left;
    dy = rect.bottom - rect.top;
    for (irgpt = 0; irgpt < 10; irgpt++)
        {
        rgrgpt[irgpt][0].x = rect.left;
        rgrgpt[irgpt][0].y = lbm.fStraight ? rect.top :
          rect.top + MulDiv((irgpt + 1), dy, 11);
        rgrgpt[irgpt][1].x = lbm.fStraight ?
          ((irgpt & 1) ? rect.left : rect.right) :
            rect.left + MulDiv((irgpt + 1), dx, 11);
        rgrgpt[irgpt][1].y =
          !lbm.fStraight || (irgpt & 1) ?
            rect.bottom : rect.top;
        }

    InvalidateRect(hwnd, NULL, TRUE);
    UpdateWindow(hwnd);
    tim = timeGetTime();
    for (ilin = 0; ilin < lbm.cact; ilin++)
        {
        irgpt = ilin % 10;
        PenLine(hdc, rgrgpt[irgpt], lbm.dxPen,
          rgclr[ilin % 8]);
        }
    dtimPenLine = timeGetTime() - tim;
    InvalidateRect(hwnd, NULL, TRUE);
    UpdateWindow(hwnd);
    tim = timeGetTime();
    for (ilinLim = ilin + lbm.cact; ilin < ilinLim;
      ilin++)
        {
        irgpt = ilin % 10;
        FastLine(hdc, rgrgpt[irgpt], lbm.dxPen,
          rgclr[ilin % 8]);
        }
    dtimFastLine = timeGetTime() - tim;
	fInBench = FALSE;
    ReleaseDC(hwnd, hdc);
    InvalidateRect(hwnd, NULL, TRUE);
    }

void
PenLine(HDC hdc, POINT prgpt[2], int dxPen,
  COLORREF clr)
/*****************************************************/
/* -- Draw a line using MoveTo() and LineTo().       */
/* -- hdc   : Device context to draw into.           */
/* -- prgpt : End points of line.                    */
/* -- dxPen : Thickness of line.                     */
/* -- clr   : Color to draw line.                    */
/*****************************************************/
    {
    HPEN    hpen, hpenSav;

    if ((hpen = CreatePen(PS_SOLID, dxPen, clr)) ==
      NULL)
        return;

    hpenSav = SelectObject(hdc, hpen);
    MoveTo(hdc, prgpt[0].x, prgpt[0].y);
    LineTo(hdc, prgpt[1].x, prgpt[1].y);
    if (hpenSav != NULL)
        SelectObject(hdc, hpenSav);
    DeleteObject(hpen);
    }

BOOL CALLBACK __export
FLineMetricsDlgProc(HWND hwnd, UINT wm, WPARAM wParam,
  LPARAM lParam)
/*****************************************************/
/* -- Line benchmark metrics dialog box procedure.   */
/*****************************************************/
    {
    switch (wm)
        {
    default:
        break;

    case WM_INITDIALOG:
        CheckRadioButton(hwnd, didStraight,
          didDiagonal,
          lbm.fStraight ? didStraight : didDiagonal);
        SetDlgItemInt(hwnd, didThickness, lbm.dxPen,
          FALSE);
        SetDlgItemInt(hwnd, didLineCount, lbm.cact,
          FALSE);
        return TRUE;

    case WM_COMMAND:
        {
        switch (wParam)
            {
        default:
            break;

        case didStraight:
        case didDiagonal:
            switch (HIWORD(lParam))
                {
            default:
                break;

            case BN_CLICKED:
                CheckRadioButton(hwnd, didStraight,
                  didDiagonal, wParam);
                return TRUE;

            case BN_DOUBLECLICKED:
                {
                HWND    hwndOK;

                hwndOK = GetDlgItem(hwnd, IDOK);
                PostMessage(hwndOK, WM_LBUTTONDOWN,
                  0, 0);
                PostMessage(hwndOK, WM_LBUTTONUP,
                  0, 0);
                }
                return TRUE;
                }
            break;

        case IDOK:
            {
            LBM     lbmT;

            lbmT.fStraight =
              IsDlgButtonChecked(hwnd, didStraight);
            if (FGetW(&lbmT.dxPen, didThickness, hwnd)
              && FGetW(&lbmT.cact, didLineCount, hwnd))
                {
                lbm = lbmT;
                EndDialog(hwnd, TRUE);
                return TRUE;
                }
            }
            break;

        case IDCANCEL:
            EndDialog(hwnd, FALSE);
            return TRUE;
            }   /* End switch wParam. */
        }
        break;  /* End case WM_COMMAND. */
        }       /* End switch wm. */

    return FALSE;
    }

BOOL
FGetW(UINT * puw, UINT did, HWND hwnd)
/*****************************************************/
/* -- Get a whole number for the given control.      */
/*****************************************************/
    {
    BOOL    fOk;
    char    szBuf[256];

    *puw = GetDlgItemInt(hwnd, did, &fOk, FALSE);
    if (fOk && *puw != 0)
        return TRUE;

    szBuf[0] = '"';
    GetDlgItemText(hwnd, did, szBuf + 1,
      sizeof szBuf - 1);
    lstrcat(szBuf, "\"is invalid."
      "  Please enter a positive integer");
    MessageBox(hwnd, szBuf, szTitle, MB_OK);
    return FALSE;
    }

void BenchPlacement(HWND hwnd)
/*****************************************************/
/* -- Benchmark the placement of graphics along 8    */
/*    pixel boundaries.                              */
/*****************************************************/
    {
    RECT    rect;
    HDC     hdc;
    DWORD   tim;
    int     irect;
    int     x;

    if ((hdc = GetDC(hwnd)) == NULL)
        return;
	fInBench = TRUE;
    GetClientRect(hwnd, &rect);

    /* Do aligned rectangles. */
    InvalidateRect(hwnd, NULL, TRUE);
    UpdateWindow(hwnd);
    tim = timeGetTime();
    for (x = 0, irect = 0; irect < crect; irect++)
        {
        PatBlt(hdc, x, 0, 16, rect.bottom,
          (irect & 1) ? BLACKNESS : WHITENESS);
        if ((x += 8) > rect.right)
            x = 0;
        }
    dtimAligned = timeGetTime() - tim;

    /* Do unaligned rectangles. */
    rect.left++;
    rect.right++;
    InvalidateRect(hwnd, NULL, TRUE);
    UpdateWindow(hwnd);
    tim = timeGetTime();
    for (x = 1, irect = 0; irect < crect; irect++)
        {
        PatBlt(hdc, x, 0, 16, rect.bottom,
          (irect & 1) ? BLACKNESS : WHITENESS);
        if ((x += 8) > rect.right)
            x = 0;
        }
    dtimUnaligned = timeGetTime() - tim;

	fInBench = FALSE;
    ReleaseDC(hwnd, hdc);
    InvalidateRect(hwnd, NULL, TRUE);
    }
