/*****************************************************/
/* fastline.c                                        */
/* -- Routine to draw a thick line quickly.          */
/*****************************************************/

#include <windows.h>
#include "fastline.h"

UINT WSqrtLw(long lw);

void
FastLine(HDC hdc, POINT prgpt[2], int dxPen,
  COLORREF clr)
/*****************************************************/
/* -- Draw a line as quickly as possible.            */
/* -- hdc   : Device context to draw into.           */
/* -- prgpt : End points of line.                    */
/* -- dxPen : Thickness of line.                     */
/* -- clr   : Color to draw line.                    */
/*****************************************************/
    {
    HPEN    hpen    = NULL;
    HPEN    hpenSav = NULL;
    HBRUSH  hbrs    = NULL;
    HBRUSH  hbrsSav = NULL;
    int     dxHalf  = (dxPen + 1) >> 1;
    int     x, y, dx, dy;

    if (dxPen == 1)
        {
        /* Optimize for thin lines. */
        if ((hpen = CreatePen(PS_SOLID, 1, clr)) !=
          NULL)
            hpenSav = SelectObject(hdc, hpen);
        MoveTo(hdc, prgpt[0].x, prgpt[0].y);
        LineTo(hdc, prgpt[1].x, prgpt[1].y);
        }
    else if (prgpt[0].x == prgpt[1].x)
        {
        /* Optimize for vertical lines. */
        x = prgpt[0].x - dxHalf;
        dx = dxPen;
        y = prgpt[0].y;
        dy = prgpt[1].y - y;
        goto LDrawRect;
        }
    else if (prgpt[0].y == prgpt[1].y)
        {
        /* Optimize for horizontal lines. */
        x = prgpt[0].x;
        dx = prgpt[1].x - x;
        y = prgpt[0].y - dxHalf;
        dy = dxPen;

LDrawRect:
        if ((hbrs = CreateSolidBrush(clr)) != NULL)
            hbrsSav = SelectObject(hdc, hbrs);
        PatBlt(hdc, x, y, dx, dy, PATCOPY);
        }
    else
        {
        POINT   dpt, rgpt[4];
        int     dxy;

        /* Determine endpoints. */
        dpt.x = prgpt[1].x - prgpt[0].x;
        dpt.y = prgpt[1].y - prgpt[0].y;
        dxy = (int)WSqrtLw(((long)dpt.x * dpt.x) +
          ((long)dpt.y * dpt.y));
        rgpt[0].x =
          prgpt[0].x - MulDiv(dxHalf, dpt.y, dxy);
        rgpt[0].y =
          prgpt[0].y + MulDiv(dxHalf, dpt.x, dxy);
        rgpt[1].x =
          rgpt[0].x + MulDiv(dxPen, dpt.y, dxy);
        rgpt[1].y =
          rgpt[0].y - MulDiv(dxPen, dpt.x, dxy);
        rgpt[2].x = rgpt[1].x + dpt.x;
        rgpt[2].y = rgpt[1].y + dpt.y;
        rgpt[3].x = rgpt[0].x + dpt.x;
        rgpt[3].y = rgpt[0].y + dpt.y;
        if ((hbrs = CreateSolidBrush(clr)) != NULL)
            hbrsSav = SelectObject(hdc, hbrs);
        hpenSav =
          SelectObject(hdc, GetStockObject(NULL_PEN));
        Polygon(hdc, rgpt, 4);
        }

    if (hbrsSav != NULL)
        SelectObject(hdc, hbrsSav);
    if (hbrs != NULL)
        DeleteObject(hbrs);
    if (hpenSav != NULL)
        SelectObject(hdc, hpenSav);
    if (hpen != NULL)
        DeleteObject(hpen);
    }

UINT
WSqrtLw(long lw)
/*****************************************************/
/* -- Perform an integer square root operation.      */
/*****************************************************/
    {
	UINT    wHi, wLo;
	long    lwT;

    /* Find largest power of 2 greater than lw. */
    for (lwT = wHi = 1; lwT < lw; wHi <<= 1, lwT <<= 2)
        ;

	if (lwT == lw)
		return wHi;

    /* Binary search: wLo^2 < lw <= wHi^2 */
	for (wLo = wHi >> 1; wLo < wHi; )
		{
		UINT    wMid = (wLo + wHi) >> 1;

		lwT = (long)wMid * wMid;

		if (lwT < lw)
			wLo = wMid + 1;
		else if (lwT > lw)
			wHi = wMid;
		else
			return wMid;
		}

    /* Return whichever side is closest. */
	wLo = wHi - 1;
	return ((long)wHi * wHi - lw < lw - (long)wLo * wLo) ? wHi : wLo;
}
