/*-------------------------------------
 CTIMER.CPP - CTimer implementation.
-------------------------------------*/
#ifndef STRICT
#define STRICT
#endif

#include <windows.h>
#include <mmsystem.h>
#include "ctimer.h"
#include "timerdll.h"

#define         DESIRED_RESOLUTION      1               // Desired resolution (1 millisecond)

UINT    CTimer::pm_manyTimers=0;                // Initialize static member variable
UINT    CTimer::pm_nextTimerID=0;               // Initialize static member variable

static  CTimer  thePhantomTimer;                // Used to initialize the CTimer class
										// (Get rid of this line if you don't 
										// want one time class initialization)

//----------------------------------------------------------------------------
// CTimer contructor. 
//----------------------------------------------------------------------------
CTimer::CTimer()
{
TIMECAPS        timeCaps;

	//----- Initialize member variables -------
	m_bRunning=FALSE;               // Timer not running
	m_dwStartTime=0L;               // No start time
	m_dwStopTime=0L;                // No stop time
    m_dwEventID=0L;                     // No event ID (none in progress)

	//-----------------------------------------------------
	// If this is the first timer then we will perform the
	// class initialization. Unfortunately C++ does not
	// support class initialization in the language so we
	// will do it this way. 
	//
	// For a CTimer, class initialization consists of making
	// a function call to timeGetDevCaps() to find out the
	// minimum supported timer resolution. This will be two
	// (2) milliseconds on most 286's and real slow 386's.
	// Most "real" machines will return a value of one (1).
	//
	// Once we get the minimum resolution we will use the
	// greater of the minimum supported resolution and the
	// resolution we asked for (DESIRED_RESOLUTION) to
	// initialize the multimedia timer services. We also
	// call SetMinDllPeriod() to inform the DLL of the
	// minimum value we support. The DLL needs this in case
	// it must reschedule a one-shot event on it's own.
	//-----------------------------------------------------
	if(!pm_manyTimers++)    // Done for class initialization
	{
		timeGetDevCaps(&timeCaps, sizeof(TIMECAPS));
		m_minRes=max(DESIRED_RESOLUTION, timeCaps.wPeriodMin);
		SetMinDllPeriod(m_minRes);              // Inform DLL of the minimum timer period
		timeBeginPeriod(m_minRes);              // Initialize timer services
	}

	m_IdCTimer = ++pm_nextTimerID;          // Set the unique timer ID
}


//----------------------------------------------------------------------------
// CTimer destructor. 
//----------------------------------------------------------------------------
CTimer::~CTimer()
{
	//----------------------------------------------------------
	// Cancel any event which may be associated with the CTimer.
	// This is done in the DLL and non-existant events will be
	// safely ignored.
	//----------------------------------------------------------
	CancelEvent();

	if(! --pm_manyTimers)                   // When the last timer has been removed.
		timeEndPeriod(m_minRes);        // Call to end multimedia timer services.
}


//----------------------------------------------------------------------------
// CancelEvent() - Cancels any event which may be in progess. This is inlined
// and simply calls a function in the DLL which actually cancels the event.
//----------------------------------------------------------------------------
BOOL _inline CTimer::CancelEvent()
{
	return EventClear(m_dwEventID); // Invalid event ID's are checked
									// for and ignored in the DLL.
}

//----------------------------------------------------------------------------
// InEvent() - This too is handled by the DLL. We try to find a index to our
// event in the DLL's event table. If we found the index (zero based) then we
// will return TRUE since the event does exist. If the event does not exist
// then the return value to us will be -1 we we will return a FALSE.
//----------------------------------------------------------------------------
BOOL CTimer::InEvent()
{
	if(EventGetIndex(m_dwEventID) != -1)    // If we got a valid index to
		return TRUE;                                            // the event then it's valid.
	else
		return FALSE;                                           // Otherwise event is invalid.
}


//----------------------------------------------------------------------------
// EventOneShot() - This is the front end to the NewEvent() function. It is
// overloaded to accept either a callback function (this one) or a message.
// It simply cancels any event which may be in progress and then it calls
// the NewEvent() function which is in the DLL. NewEvent() is what actually
// creates the event. A failed event will return a 0L which we will return
// as a boolean FALSE.
//----------------------------------------------------------------------------
BOOL    CTimer::EventOneShot(UINT wDelay, FARPROC lpfnCallback, DWORD dwUser)
{
	CancelEvent();
    m_dwEventID = NewEvent(m_IdCTimer,
	EVTBIT_ONESHOT | EVTBIT_CALLBACK,
	wDelay,
	(DWORD)lpfnCallback,
	dwUser);

return (BOOL) m_dwEventID;
}

//----------------------------------------------------------------------------
// EventOneShot() - This is the front end to the NewEvent() function. It is
// overloaded to accept either a message (this one) or a callback.
// It simply cancels any event which may be in progress and then it calls
// the NewEvent() function which is in the DLL. NewEvent() is what actually
// creates the event. A failed event will return a 0L which we will return
// as a boolean FALSE.
//----------------------------------------------------------------------------
BOOL    CTimer::EventOneShot(UINT wDelay, HWND hParent, UINT idMsg, DWORD dwUser)
{
	CancelEvent();
    m_dwEventID = NewEvent(m_IdCTimer,
	EVTBIT_ONESHOT | EVTBIT_MESSAGE,
	wDelay, MAKELONG(hParent,idMsg),
	dwUser);
return (BOOL) m_dwEventID;
}

//----------------------------------------------------------------------------
// EventPeriodic() - This is the front end to the NewEvent() function. It is
// overloaded to accept either a callback function (this one) or a message.
// It simply cancels any event which may be in progress and then it calls
// the NewEvent() function which is in the DLL. NewEvent() is what actually
// creates the event. A failed event will return a 0L which we will return
// as a boolean FALSE.
//----------------------------------------------------------------------------
BOOL    CTimer::EventPeriodic(UINT wDelay, FARPROC lpfnCallback, DWORD dwUser)
{
	CancelEvent();
    m_dwEventID = NewEvent(m_IdCTimer,
	EVTBIT_PERIODIC | EVTBIT_CALLBACK,
	wDelay, (DWORD)lpfnCallback,
	dwUser);
return (BOOL) m_dwEventID;
}

//----------------------------------------------------------------------------
// EventPeriodic() - This is the front end to the NewEvent() function. It is
// overloaded to accept either a message (this one) or a callback.
// It simply cancels any event which may be in progress and then it calls
// the NewEvent() function which is in the DLL. NewEvent() is what actually
// creates the event. A failed event will return a 0L which we will return
// as a boolean FALSE.
//----------------------------------------------------------------------------
BOOL    CTimer::EventPeriodic(UINT wDelay, HWND hParent, UINT idMsg, DWORD dwUser)
{
	CancelEvent();
    m_dwEventID = NewEvent(m_IdCTimer,
	EVTBIT_PERIODIC | EVTBIT_MESSAGE,
	wDelay, MAKELONG(hParent,idMsg),
	dwUser);
return (BOOL) m_dwEventID;
}

