//---------------------------------------------------------
// Timerdll.c
//
//     Provides support for CTimer events. The DLL provides
// the event interrupt handler for CTimer events. The handler
// is provided here since all code and data which is the
// event handler references must be in segments with
// fixed and nondiscardable segment attributes.
//---------------------------------------------------------
#ifndef STRICT                          // Make sure we are using STRICT
#define STRICT
#endif

#include    <windows.h>                 // Basic definitions for windows
#include    <mmsystem.h>                // Multimedia serives 
#include    "timerdll.h"                // Our own definitions

	//-------------------------------------------------
	// Windows (actually mmsystem) supports a maximum
	// of eight events which may be outstanding at any
	// one time. We save space in the DLL by limiting
	// the event table to the same amount.
	//-------------------------------------------------
#define MAX_EVENTS 8

	//-------------------------------------------------
	// Template for the event table structure. The table
	// has the following entires in it.
	//-------------------------------------------------
typedef struct tagMYEVENT{
    UINT    idCTimer;                           // ID of the CTimer who owns this event
    UINT    idEvent;                            // ID of the event (from timeSetEvent())
    DWORD   dwNotify;                           // Who to notify. Either a callback function
								// or LOWORD/HIWORD Window and Message (see wType)
    DWORD   dwUser;                                     // 32-bits of user supplied data.
    WORD    wMissed;                            // Number of missed calls to PostMessage()
    WORD    wType;                                      // Set bits determine the type of event this is.
}MYEVENT;                                                       // The values are defined in timerdll.h and they
									// tell us if this is a ONE_SHOT or PERIODIC event.
									// The bits also tell us if this is a CALLBACK type
									// of event or if we should send (actually "post") a
									// message to a window. If this is a CALLBACK type of
									// event then dwNotify is the address of the callback
									// function. Otherwise dwNotify is a HIWORD/LOWORD with
									// the window handle and message type to send.


static UINT MinimumResolution=5;        // Static variable. Initialized to (5) milliseconds
									// although SetMinDllPeriod() will initialize it
									// to the real minimum value.

static MYEVENT EventTable[MAX_EVENTS];  // Event table to hold all of the events which
										// are outstanding at any given time.

//---------------------------------------------------------
// This is the function which actually creates events. Calls
// to EventOneShot() and EventPeriodic() get sent here where
// the actual event gets created.
//---------------------------------------------------------
DWORD WINAPI NewEvent(UINT idCTimer,
	       UINT wType, UINT wDelay,
	       DWORD dwNotify, DWORD dwUser)
{
int i;

    for(i=0; i<MAX_EVENTS; i++)                 // Look for an empty event in the event table
	if(!EventTable[i].idEvent)              // If the event ID is zero (0) then we found one.
	    break;

    if(i==MAX_EVENTS)                                   // If we broke out of the loop without finding a
	return 0L;                                              // empty event then return zero (failure)

    //-----------------------------------------------------
    // Set up the fields in the EventTable struture.
    //-----------------------------------------------------
    EventTable[i].idCTimer=idCTimer;    // The CTimer who owns this event
    EventTable[i].wMissed=0;                    // Missed calls to PostMessage()
    EventTable[i].wType=wType;                  // Type of event (1-shot/periodic & callback/messsage)
    EventTable[i].dwNotify=dwNotify;    // Who to notify (callback function or window/message)
    EventTable[i].dwUser=dwUser;                // 32 bits of user supplied data
    
    //-----------------------------------------------------
    // Call timeSetEvent() to actually create the event.
    // Store the ID of the event in the event table.
    //-----------------------------------------------------
    EventTable[i].idEvent=timeSetEvent(wDelay,
	1, TimeFunc, (DWORD)i,
	(wType & EVTBIT_ONESHOT) ?
	TIME_ONESHOT : TIME_PERIODIC);

    if(!EventTable[i].idEvent)                  // If the event ID is zero then we failed. Return
	return 0L;                                              // a zero (0) to indicate that we failed.

		//----------------------------------------------------------
		// Otherwise combine the eventID and the CTimerID to form a
		// unique identifier for this one event.
		//----------------------------------------------------------
return (DWORD)MAKELONG(EventTable[i].idEvent,idCTimer);
}

//----------------------------------------------------------
// Find the index of a event in the EventTable. The index is
// zero based. If the event can't be found then the return
// value is -1. Possible reasons that the event cannot be
// found is that the dwEventID is garbage or the event
// may have already occurred (one-shot events).
//----------------------------------------------------------
int WINAPI EventGetIndex(DWORD dwEventID)
{
int     i;
UINT    idCTimer;
UINT    idEvent;

    if(!dwEventID)              // If the dwEvent is zero then we automatically know
	return -1;              // that it is garbage so return failure.

    idCTimer=HIWORD(dwEventID); // Seperate the event ID into its two basic
    idEvent=LOWORD(dwEventID);  // components. The CTimer ID and the EventID.
    
    for(i=0; i< MAX_EVENTS; i++)        // Loop through the entire event table
    {
	if(EventTable[i].idEvent == idEvent &&
		EventTable[i].idCTimer == idCTimer)
	    return i;                           // If we found a match then return the index
    }

return -1;                                                      // Otherwise the event no longer exists.
}

//----------------------------------------------------------
// This is somewhat of an oddball function. It is called by
// the CTimer class during class initialization. It informs
// the DLL what the minimum supported time interval is. The
// only reason that the DLL needs to know this is the case
// where a 1-shot event occurs and we can't post a message
// to the window. We need to create another 1-shot event
// and try PostMessage() again. We create the new 1-shot
// event so it goes off as soon as possible. So we don't
// fail in creating the event, we know that the minimum
// allowable time for the 1-shot event is minRes.
//----------------------------------------------------------
void WINAPI SetMinDllPeriod(UINT minRes)
{
    MinimumResolution=minRes;           // Save the minimum allowable event time
}


//----------------------------------------------------------
// Clears an event. This is the actual code behind the CTimer
// member function CTimer::CancelEvent(). 
//----------------------------------------------------------
BOOL WINAPI EventClear(DWORD  dwEventID)
{
int     index;
UINT    status;

    index=EventGetIndex(dwEventID);             // Find the event in the event table
    
    if(index != -1)                                             // If the event is a valid one......
    {
	status= timeKillEvent(LOWORD(dwEventID));       // Kill the event and
	EventTable[index].idEvent=0;                            // zero out the event fields
	EventTable[index].idCTimer=0;                           // so we don't use them again.
	
	if(!status)                                             // Status will be zero if we were able to cancel
	    return TRUE;                                // the event. Failure could have resulted if the
    }                                                                   // event interrupt occured while we were in the
										// middle of processing this function.

return FALSE;                                                   // We couldn't cancel the event.
}


//----------------------------------------------------------
//             I N T E R R U P T  -  C O D E
//
// This is the big one. This is the function which is called
// at interrupt time when a timer event goes off. I will walk
// through this step by step and describe what is going on.
//----------------------------------------------------------
void CALLBACK TimeFunc(UINT wTimerID,
	      UINT wMsg, DWORD dwUser,
	      DWORD dw1, DWORD dw2)
{
int i;
void (FAR PASCAL *pFunc)(DWORD);

    i=(int) dwUser;             // The value of 'i' tells us the index of this event
					// in the event table. When we first created the event
					// we set dwUser to be the value of 'i' which is it's
					// index in the event table. The allows us not to have
					// to search through the EventTable at interrupt time.

	//-------------------------------------------------------------------------
	// First we will handle the simple case. If the CALLBACK bit is set then
	// we know that the event needs to call the callback function which is
	// stored as dwNotify. If this is the case then we will call the callback
	// function and pass the dwUser value to it.
	//-------------------------------------------------------------------------
    if(EventTable[i].wType & EVTBIT_CALLBACK)
    {
	pFunc=(void far *) EventTable[i].dwNotify;
	(*pFunc)(EventTable[i].dwUser);
    }
	//-------------------------------------------------------------------------
	// 'else' means that it is not a callback function so we need to post a 
	// message to a window. The tricky part is that the we may not be able to
	// post the message successfully. Let's see how we do this:
	//-------------------------------------------------------------------------
    else
    {
	//--------------------------------------------------------------------
	// Try to post the message. The function really looks like this when
	// cleaned up:
	//        PostMessage(hWnd, wMsg, wMissed, dwUser);
	//--------------------------------------------------------------------
	if(!PostMessage((HWND)LOWORD(EventTable[i].dwNotify),
		    (UINT)HIWORD(EventTable[i].dwNotify),
		    (WPARAM)EventTable[i].wMissed,
		    (LPARAM) EventTable[i].dwUser))
	    EventTable[i].wMissed++;    // PostMessage() failed. Increment the "Missed" count.
	else
	    EventTable[i].wMissed=0;    // PostMessage() worked. Clear "Missed" count to zero.
    }

	//-------------------------------------------------------------------------
	// At this point we are in the following state:
	//
	// If the event required a callback then we already called the function.
	// If the event required a message then we tried to post it. If the call
	//    to PostMessage() worked wMissed is zero. Otherwise wMissed is the
	//    total number of PostMessage() calls that failed.
	//-------------------------------------------------------------------------


	//-------------------------------------------------------------------------
	// We now want to make sure that the window really did get the message. If
	// it is a PERIODIC event then we don't really have to worry since we will
	// get another chance at PostMessage() the next time the event occurs.
	//-------------------------------------------------------------------------
    if(EventTable[i].wType & EVTBIT_PERIODIC)           // If it's a periodic event
	return;                                                                         // then return.

	//-------------------------------------------------------------------------
	// We are now at the last case. If the event was a callback then 'wMissed'
	// will always be zero. If it is not zero then we know that the event was
	// a ONE_SHOT event and we couldn't post the message. We need to create
	// another event and then try PostMessage() later when that event occurs.
	// We will keep trying this until PostMessage() finally works.
	//-------------------------------------------------------------------------
    
    if(!EventTable[i].wMissed)             // PostMessage() to the 1-shot event WORKED
    {
	EventTable[i].idEvent=0;           // Clear the values in the event table so
	EventTable[i].idCTimer=0;          // we can use them again.
	return;                            // Return happily.
    }

	//-------------------------------------------------------------------------
	// Create another 1-shot event and store the info on the event
	// in the same index in the event table. The call to timeSetEvent()
	// should work since this was a 1-shot event which we are processing.
	// This means that the mmsystem should be able to create another event
	// since nobody else would have had a chance to create one before us.
	// Note that is is OK to call timeSetEvent() at interrupt time.
	//-------------------------------------------------------------------------
    EventTable[i].idEvent=timeSetEvent(MinimumResolution,
				1, TimeFunc, dwUser, TIME_ONESHOT);

return;
}

