/*****************************************************
           File Name: EDITRANG.C
         Description: Library of functions for
                      edit range custom control
Global Function List: EditRangeProc
Static Function List: IsCharType
         Portability: MS Windows, Any memory model,
                      Any windows compatable C Compiler
******************************************************/

#include <ctype.h>
#include <string.h>
#include <windows.h>
#include "subclass.h"
#include "editrang.h"

/* Static function prototypes */
static int IsCharType( int Char, WORD Type );

/*****************************************************
       Name: EditRangeProc
Description: Window proc for edit range custom control
             window class.
*****************************************************/
LONG CALLBACK EditRangeProc( HWND hWnd,
      UINT wMessage, WPARAM wParam, LPARAM lParam )
   {
   BOOL CharInRange;

   if ( wMessage == WM_CREATE )
      {
      WORD Style;
      LONG SpecialChars;

      /* Set the default style that is in the class
      ** extra bytes. */
      Style = GetSubClassWord( hWnd, OFFSET_STYLE );
      SetSubWindowWord( hWnd, OFFSET_STYLE, Style );
      SpecialChars = GetSubClassLong( hWnd,
            OFFSET_SPECIALCHARS );
      SetSubWindowLong( hWnd, OFFSET_SPECIALCHARS,
            SpecialChars );

      }   /* if wMessage */

   if ( wMessage == WM_CHAR )
      {
      int AsciiChar;
      WORD Style, Type;
   
      AsciiChar = (int)wParam;

      /* Retrieve the style from the window
      ** property list */
      Style = GetSubWindowWord( hWnd, OFFSET_STYLE );

      if ( Style & ERS_TOLOWER )
         {
         /* Convert the charater to lower case */
         AsciiChar = tolower( AsciiChar );
         wParam = (WORD)AsciiChar;
         }

      if ( Style & ERS_TOUPPER )
         {
         /* Convert the charater to upper case */
         AsciiChar = toupper( AsciiChar );
         wParam = (WORD)AsciiChar;
         }

      CharInRange = FALSE;
      Type = ERS_ALNUM;
      while ( !CharInRange && Type <= ERS_XDIGIT )
         {
         /* Loop through all the style types */
         if ( Style & Type )
            {
            /* Check if character is this type */
            CharInRange = IsCharType( AsciiChar,
                  Type );
            }
         Type <<= 1;
         }   /* while */

      if ( !CharInRange && ( Style & ERS_SPECIAL ) )
         {
         LPSTR SpecialChars;

         /* Get the SpecialChars pointer from the
         ** window property list. */
         SpecialChars = (LPSTR)GetSubWindowLong(
               hWnd, OFFSET_SPECIALCHARS );
         if ( ( SpecialChars != NULL ) &&
               ( _fstrchr( SpecialChars,
               (char)AsciiChar ) != NULL ) )
            CharInRange = TRUE;
         }   /* if !CharInRange */

      if ( !CharInRange )
         {
         if ( Style & ERS_BEEP )
            MessageBeep( 0 );
         if ( Style & ERS_MESSAGE )
            {
            MessageBox( hWnd, "You are attempting "
                  "to enter characters that "
                  "are outside the valid range "
                  "for this edit field.", NULL,
                  MB_ICONINFORMATION | MB_OK );
            SetFocus( hWnd );
            }
         }   /* if !CharInRange */

      }   /* if wMessage */
   if ( CharInRange || wMessage != WM_CHAR )
      {
      FARPROC EditWndProc;

      /* Pass message to the parent cls wnd proc */
      EditWndProc = GetParentClsWndProc( hWnd );
      if ( EditWndProc != NULL )
         return ( CallWindowProc( (WNDPROC)EditWndProc,
               hWnd, wMessage, wParam,
               lParam ) );
      }   /* if CharInRange */
   return ( TRUE );
   }   /* function EditCharRangeProc */

/*****************************************************
       Name: IsCharType
 Parameters: Char - ASCII character
             Type - type of character to check for
     Return: TRUE if character type match else FALSE
Description: Checks to see if a character is a
             specified type.
*****************************************************/
static int IsCharType( int Char, WORD Type )
   {
   switch ( Type )
      {
      case ERS_ALNUM: return( isalnum( Char ) );
      case ERS_ALPHA: return( isalpha( Char ) );
      case ERS_CNTRL: return( iscntrl( Char ) );
      case ERS_DIGIT: return( isdigit( Char ) );
      case ERS_GRAPH: return( isgraph( Char ) );
      case ERS_LOWER: return( islower( Char ) );
      case ERS_PRINT: return( isprint( Char ) );
      case ERS_PUNCT: return( ispunct( Char ) );
      case ERS_SPACE: return( isspace( Char ) );
      case ERS_UPPER: return( isupper( Char ) );
      case ERS_XDIGIT: return( isxdigit( Char ) );
      default: return ( FALSE );
      }
   }   /* function IsCharType */
