/*****************************************************
           File Name: SUBCLASS.C
         Description: Library of functions for
                      subclassing windows.
Global Function List: RegisterSubClass, GetParentClsWndProc
                      GetSubClsIndex, GetSubWndIndex
                      GetSubClassLong, GetSubClassWord
                      GetSubWindowLong, GetSubWindowWord
                      SetSubClassLong, SetSubClassWord
                      SetSubWindowLong, SetSubWindowWord
    Local Macro List: PARENT_PROC_INDEX
                      PARENT_CLSEXTRA_INDEX
                      PARENT_WNDEXTRA_INDEX
         Portability: MS Windows, Any memory model,
                      Any windows compatable C Compiler
******************************************************/

#include <windows.h>
#include "subclass.h"

/* Offsets of parent class window information
** from the end of the class extra bytes of the
** subclass window class structure */
#define PARENT_PROC_INDEX ( sizeof( FARPROC ) )
#define PARENT_CLSEXTRA_INDEX \
      ( sizeof( FARPROC ) + sizeof( int ) )
#define PARENT_WNDEXTRA_INDEX \
      ( sizeof( FARPROC ) + sizeof( int ) \
      + sizeof( int ) )

/*****************************************************
       Name: RegisterSubClass
 Parameters: SubClsName - string containing name of
                          subclass window class
             ParClsName - string containing name of
                          parent window class
             hInstance - handle of the instance
                         registering this subclass
             FARPROC - pointer to window proc for
                       the subclass windows
             SubClsClsExtra - number of class extra
                              bytes for the subclass
             SubClsWndExtra - number of window extra
                              bytes for the subclass
     Return: TRUE for success, FALSE for failure
Description:
    Loads a WNDCLASS structure with information contained
    in an existing window class (Parent Class).  This
    structure is modified to subclass window values.  This
    new subclass window class is then registered.  Note -
    the values of style, hIcon, hCursor, hbrBackground and
    lpszMenuName remain the same as the parent class. 
    Subclasses that require modification to these values
    can not use this functon and require individual
    treatment. 
*****************************************************/
BOOL FAR PASCAL RegisterSubClass( char *SubClsName,
      char *ParentClsName, HINSTANCE hInstance,
      WNDPROC SubClsWndProc,
      int SubClsClsExtra, int SubClsWndExtra )
   {
   HWND hWnd;
   WNDCLASS WndClass;

   /* Load WndClass with parent class info */
   if ( !GetClassInfo( NULL, ParentClsName,
         &WndClass ) )
      /* Could not find the parent class */
      return ( FALSE );

   /* Set class values to subclass info */
   WndClass.cbClsExtra += (int)PARENT_WNDEXTRA_INDEX
         + SubClsClsExtra;
   WndClass.cbWndExtra += SubClsWndExtra;
   WndClass.hInstance = hInstance;
   WndClass.lpszClassName = SubClsName;

   /* Regsiter the new subclass */
   if ( !RegisterClass( &WndClass ) )
      /* Could not register the new class */
      return ( FALSE );
   hWnd = CreateWindow( SubClsName, "", 0,
         CW_USEDEFAULT, CW_USEDEFAULT,
         CW_USEDEFAULT, CW_USEDEFAULT,
         NULL, NULL, hInstance, NULL );
   if ( hWnd == NULL )
      {
      /* Could not create a window */
      UnregisterClass( WndClass.lpszClassName,
            WndClass.hInstance );
      return ( FALSE );
      }

   /* Store the parent class info in the sub
   ** class class extra bytes */
   SubClsClsExtra = WndClass.cbClsExtra;
   GetClassInfo( NULL, ParentClsName, &WndClass );
   SetClassLong( hWnd, (int)SubClsClsExtra -
         (int)PARENT_PROC_INDEX,
         (LONG)WndClass.lpfnWndProc );
   SetClassWord( hWnd, (int)SubClsClsExtra -
         (int)PARENT_CLSEXTRA_INDEX,
         (WORD)WndClass.cbClsExtra );
   SetClassWord( hWnd, (int)SubClsClsExtra -
         (int)PARENT_WNDEXTRA_INDEX,
         (WORD)WndClass.cbWndExtra );
   /* Set the new window proc for the class */
   SetClassLong( hWnd, GCL_WNDPROC,
         (LONG)SubClsWndProc );
   DestroyWindow( hWnd );
   return ( TRUE );
   }   /* RegisterSubClass */

/*****************************************************
       Name: GetParentClsWndProc
 Parameters: hWnd - handle of sub classed window
     Return: pointer to parent class window proc
Description: obtained the parent window proc of a
             subclassed window
*****************************************************/
FARPROC FAR PASCAL GetParentClsWndProc( HWND hWnd )
   {
   int Index;

   Index = (int)GetClassWord( hWnd, GCW_CBCLSEXTRA );
   Index -= (int)PARENT_PROC_INDEX;
   return ( (FARPROC)GetClassLong( hWnd, Index ) );
   }   /* function GetParentClsWndProc */

/*****************************************************
       Name: GetSubClsIndex
 Parameters: hWnd - handle of subclassed window
             Index - offset of subclassed extra byte
     Return: Extra byte offset index
Description: Gets the index, byte offset, of class
             extra byte information.
*****************************************************/
int GetSubClsIndex( HWND hWnd, int Index )
   {
   if ( Index < 0 )
      /* Offset is into window structure
      ** members so return unchanged */
      return ( Index );
   /* Get the size of the parent class cls extra
   ** bytes add to the index and return */
   return ( Index + (int)GetClassWord( hWnd,
         (int)GetClassWord( hWnd, GCW_CBCLSEXTRA )
         - (int)PARENT_CLSEXTRA_INDEX ) );
   }   /* funciton GetSubClsIndex */

/*****************************************************
       Name: GetSubWndIndex
 Parameters: hWnd - handle of subclassed window
             Index - offset of subclassed extra byte
     Return: Extra byte offset index
Description: Gets the index, byte offset, of window 
             extra byte information.
*****************************************************/
int GetSubWndIndex( HWND hWnd, int Index )
   {
   if ( Index < 0 )
      /* Offset is into window structure
      ** members so return unchanged */
      return ( Index );
   /* Get the size of the parent class wnd extra
   ** bytes add to the index and return */
   return ( Index + (int)GetClassWord( hWnd,
         (int)GetClassWord( hWnd, GCW_CBCLSEXTRA )
         - (int)PARENT_WNDEXTRA_INDEX ) );
   }   /* funciton GetSubWndIndex */

/*****************************************************
       Name: GetSubClassLong - SetSubWindowWord
Description: The following eight functions are
             analogous to the eight Windows functions
             GetClassLong - SetWindowWord.
             The difference is that these funcitons
             are required for accessing and setting
             extra bytes of subclassed windows.
*****************************************************/
LONG FAR PASCAL GetSubClassLong( HWND hWnd,
      int nIndex )
   {
   nIndex = GetSubClsIndex( hWnd, nIndex );
   return( GetClassLong( hWnd, nIndex ) );
   }   /* function */
WORD FAR PASCAL GetSubClassWord( HWND hWnd,
      int nIndex )
   {
   nIndex = GetSubClsIndex( hWnd, nIndex );
   return( GetClassWord( hWnd, nIndex ) );
   }   /* function GetSubClassWord */
LONG FAR PASCAL GetSubWindowLong( HWND hWnd,
      int nIndex )
   {
   nIndex = GetSubWndIndex( hWnd, nIndex );
   return( GetWindowLong( hWnd, nIndex ) );
   }   /* function GetSubWindowLong */
WORD FAR PASCAL GetSubWindowWord( HWND hWnd,
      int nIndex )
   {
   nIndex = GetSubWndIndex( hWnd, nIndex );
   return( GetWindowWord( hWnd, nIndex ) );
   }   /* function GetSubWindowWord */
LONG FAR PASCAL SetSubClassLong( HWND hWnd,
      int nIndex, LONG NewLong )
   {
   nIndex = GetSubClsIndex( hWnd, nIndex );
   return( SetClassLong( hWnd, nIndex, NewLong ) );
   }   /* function SetSubClassLong */
WORD FAR PASCAL SetSubClassWord( HWND hWnd,
      int nIndex, WORD wNewWord )
   {
   nIndex = GetSubClsIndex( hWnd, nIndex );
   return( SetClassWord( hWnd, nIndex, wNewWord ) );
   }   /* funciton SetSubClassWord */
LONG FAR PASCAL SetSubWindowLong( HWND hWnd,
      int nIndex, LONG NewLong )
   {
   nIndex = GetSubWndIndex( hWnd, nIndex );
   return( SetWindowLong( hWnd, nIndex, NewLong ) );
   }   /* function SetSubWindowLong */
WORD FAR PASCAL SetSubWindowWord( HWND hWnd,
      int nIndex, WORD wNewWord )
   {
   nIndex = GetSubWndIndex( hWnd, nIndex );
   return( SetWindowWord( hWnd, nIndex, wNewWord ) );
   }   /* SetSubWindowWord */

